# PowerShell Script to Check Bitness of Microsoft Office and Access Database Engine
# Author: GitHub Copilot
# Created: May 14, 2025
# Purpose: Determine if Microsoft Office and Access Database Engine are 32-bit or 64-bit

Write-Host "=========================================================="
Write-Host "   Microsoft Office and Access Database Engine Checker    "
Write-Host "=========================================================="
Write-Host ""

# Check OS architecture
$osArch = if ([Environment]::Is64BitOperatingSystem) { "64-bit" } else { "32-bit" }
Write-Host "Operating System: $osArch Windows"

# Check PowerShell process architecture
$psArch = if ([Environment]::Is64BitProcess) { "64-bit" } else { "32-bit" }
Write-Host "PowerShell Process: $psArch"
Write-Host ""

# Check Microsoft Office installation
Write-Host "Checking Microsoft Office installations..."

$officeLocations = @(
    "HKLM:\SOFTWARE\Microsoft\Office",
    "HKLM:\SOFTWARE\Wow6432Node\Microsoft\Office"
)

$officeFound = $false

foreach ($location in $officeLocations) {
    if (Test-Path $location) {
        $officeArch = if ($location -like "*Wow6432Node*") { "32-bit" } else { "64-bit" }
        
        # Check for version specific paths
        $versions = @("16.0", "15.0", "14.0", "12.0", "11.0")
        
        foreach ($version in $versions) {
            $versionPath = Join-Path $location $version
            if (Test-Path $versionPath) {
                $officeFound = $true
                $versionMap = @{
                    "16.0" = "Office 2016/2019/2021/365"
                    "15.0" = "Office 2013"
                    "14.0" = "Office 2010"
                    "12.0" = "Office 2007"
                    "11.0" = "Office 2003"
                }
                $versionLabel = $versionMap[$version]
                Write-Host "  - Found $officeArch $versionLabel (Version $version)"
            }
        }
    }
}

if (-not $officeFound) {
    Write-Host "  - No Microsoft Office installation detected in registry"
}

Write-Host ""

# Check Access Database Engine installation
Write-Host "Checking Microsoft Access Database Engine installation..."

$aceLocations = @(
    @{ Path = "HKLM:\SOFTWARE\Microsoft\Office\16.0\Access Connectivity Engine"; Bitness = "64-bit"; Name = "ACE 16.0" },
    @{ Path = "HKLM:\SOFTWARE\Microsoft\Office\15.0\Access Connectivity Engine"; Bitness = "64-bit"; Name = "ACE 15.0" },
    @{ Path = "HKLM:\SOFTWARE\Microsoft\Office\14.0\Access Connectivity Engine"; Bitness = "64-bit"; Name = "ACE 14.0" },
    @{ Path = "HKLM:\SOFTWARE\Microsoft\Office\12.0\Access Connectivity Engine"; Bitness = "64-bit"; Name = "ACE 12.0" },
    @{ Path = "HKLM:\SOFTWARE\Wow6432Node\Microsoft\Office\16.0\Access Connectivity Engine"; Bitness = "32-bit"; Name = "ACE 16.0" },
    @{ Path = "HKLM:\SOFTWARE\Wow6432Node\Microsoft\Office\15.0\Access Connectivity Engine"; Bitness = "32-bit"; Name = "ACE 15.0" },
    @{ Path = "HKLM:\SOFTWARE\Wow6432Node\Microsoft\Office\14.0\Access Connectivity Engine"; Bitness = "32-bit"; Name = "ACE 14.0" },
    @{ Path = "HKLM:\SOFTWARE\Wow6432Node\Microsoft\Office\12.0\Access Connectivity Engine"; Bitness = "32-bit"; Name = "ACE 12.0" }
)

$aceFound = $false

foreach ($ace in $aceLocations) {
    if (Test-Path $ace.Path) {
        $aceFound = $true
        Write-Host "  - Found $($ace.Bitness) $($ace.Name)"
    }
}

# Also check via the uninstall keys which works better for standalone installs
Write-Host ""
Write-Host "Checking via Uninstall registry keys..."

$uninstallLocations = @(
    @{ Path = "HKLM:\SOFTWARE\Microsoft\Windows\CurrentVersion\Uninstall"; Bitness = "64-bit" },
    @{ Path = "HKLM:\SOFTWARE\Wow6432Node\Microsoft\Windows\CurrentVersion\Uninstall"; Bitness = "32-bit" }
)

$aceFoundInUninstall = $false

foreach ($location in $uninstallLocations) {
    $keys = Get-ItemProperty ($location.Path + "\*") -ErrorAction SilentlyContinue | 
            Where-Object { $_.DisplayName -like "*Access Database Engine*" } |
            Select-Object DisplayName, DisplayVersion

    foreach ($key in $keys) {
        $aceFoundInUninstall = $true
        Write-Host "  - Found $($location.Bitness) $($key.DisplayName) (Version $($key.DisplayVersion))"
    }
}

# Final check for the MSAccess.exe executable if Office is present
$possiblePaths = @(
    "${env:ProgramFiles}\Microsoft Office\root\Office16\MSACCESS.EXE",
    "${env:ProgramFiles(x86)}\Microsoft Office\root\Office16\MSACCESS.EXE",
    "${env:ProgramFiles}\Microsoft Office\Office16\MSACCESS.EXE",
    "${env:ProgramFiles(x86)}\Microsoft Office\Office16\MSACCESS.EXE",
    "${env:ProgramFiles}\Microsoft Office\root\Office15\MSACCESS.EXE",
    "${env:ProgramFiles(x86)}\Microsoft Office\root\Office15\MSACCESS.EXE",
    "${env:ProgramFiles}\Microsoft Office\Office15\MSACCESS.EXE",
    "${env:ProgramFiles(x86)}\Microsoft Office\Office15\MSACCESS.EXE",
    "${env:ProgramFiles}\Microsoft Office\Office14\MSACCESS.EXE",
    "${env:ProgramFiles(x86)}\Microsoft Office\Office14\MSACCESS.EXE"
)

Write-Host ""
Write-Host "Checking for Access executable..."
foreach ($path in $possiblePaths) {
    if (Test-Path $path) {
        $fileInfo = Get-Item $path
        $bitness = if ((Get-Command Get-ProcessArchitecture -ErrorAction SilentlyContinue)) {
            (Get-ProcessArchitecture $path)
        } else {
            # Check PE header manually to determine bitness
            $stream = New-Object System.IO.FileStream($path, [System.IO.FileMode]::Open, [System.IO.FileAccess]::Read)
            try {
                $reader = New-Object System.IO.BinaryReader($stream)
                $stream.Position = 0x3C
                $peOffset = $reader.ReadInt32()
                $stream.Position = $peOffset + 4
                $machineType = $reader.ReadInt16()
                
                switch ($machineType) {
                    0x014c { "32-bit" }
                    0x8664 { "64-bit" }
                    default { "Unknown" }
                }
            } finally {
                $reader.Close()
                $stream.Close()
            }
        }
        
        Write-Host "  - Found MS Access executable: $path ($bitness)"
    }
}

if (-not ($aceFound -or $aceFoundInUninstall)) {
    Write-Host "  - No Microsoft Access Database Engine detected in registry"
}

Write-Host ""
Write-Host "Provider DLL Information:"
$providerPaths = @(
    @{ Path = "${env:SystemRoot}\System32\msjetoledb40.dll"; Name = "Microsoft Jet OLEDB 4.0" },
    @{ Path = "${env:SystemRoot}\System32\msrd3x40.dll"; Name = "Microsoft Jet Red 3.x" },
    @{ Path = "${env:SystemRoot}\SysWOW64\msjetoledb40.dll"; Name = "Microsoft Jet OLEDB 4.0 (32-bit)" },
    @{ Path = "${env:SystemRoot}\SysWOW64\msrd3x40.dll"; Name = "Microsoft Jet Red 3.x (32-bit)" },
    @{ Path = "${env:SystemRoot}\System32\aceoledb.dll"; Name = "Microsoft ACE OLEDB" },
    @{ Path = "${env:SystemRoot}\SysWOW64\aceoledb.dll"; Name = "Microsoft ACE OLEDB (32-bit)" }
)

foreach ($provider in $providerPaths) {
    if (Test-Path $provider.Path) {
        $fileVersion = (Get-Item $provider.Path).VersionInfo.FileVersion
        Write-Host "  - $($provider.Name): Found - Version $fileVersion"
    } else {
        Write-Host "  - $($provider.Name): Not found"
    }
}

Write-Host ""
Write-Host "=========================================================="
Write-Host "                     Recommendations                      " 
Write-Host "=========================================================="
Write-Host ""
Write-Host "If you're running a 64-bit version of Windows:"
Write-Host "1. Ensure the bitness of Microsoft Access Database Engine matches the bitness of your Office installation"
Write-Host "2. If using a 64-bit process to access data, you need 64-bit providers"
Write-Host "3. If using a 32-bit process to access data, you need 32-bit providers"
Write-Host ""
Write-Host "For your VBScript issue:"
Write-Host "1. If running in 64-bit mode, either:"
Write-Host "   - Install the 64-bit Access Database Engine (if it doesn't conflict with Office bitness)"
Write-Host "   - Force the script to run in 32-bit mode using 'cscript /nologo /32'"
Write-Host "2. If running in 32-bit mode and still having issues, check permissions to the Access database"
Write-Host ""
Write-Host "NOTE: It's generally recommended that Office and Access Database Engine have the same bitness."
Write-Host "      Installing mismatched bitness may cause conflicts or require special installation parameters."
